/**
 * \file: NaviStatusChannel.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: AAuto
 *
 * \author: M. Adachi / ADITJ/SW / madachi@jp.adit-jv.com
 *
 * \copyright (c) 2013-2016 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#include <adit_logging.h>

#include "NaviStatusChannel.h"
#include "DemoFactory.h"

LOG_IMPORT_CONTEXT(demo)

#define NAVI_INTERVAL_MS    100
#define NAVI_IMAGE_HEIGHT   320
#define NAVI_IMAGE_WIDTH    240
#define NAVI_IMAGE_COLOR    256

namespace adit { namespace aauto {


NaviStatusChannel::NaviStatusChannel(::shared_ptr<GalReceiver> inReceiver,uint8_t inSessionId,SensorSourceChannel* inSensor)
                                      : mSessionId(inSessionId), mReceiver(inReceiver),mSensor(inSensor)
{
    LOGD_DEBUG((demo, "Navi Status Channel Constructor"));
    mEndpoint = nullptr;
}

NaviStatusChannel::~NaviStatusChannel()
{
    LOGD_DEBUG((demo, "Navi Status Channel destructor"));
    mEndpoint = nullptr;
}

void NaviStatusChannel::shutdown()
{
    LOGD_DEBUG((demo, "Navi Status Channel shut down"));
    if(mEndpoint != nullptr) {
        mEndpoint->shutdown();
    }
}
bool NaviStatusChannel::Initialize()
{
    /* This pointer is deleted at shared_ptr in impl */
    auto creator = DemoFactory::instance().getCreator
                       <AditNavigationStatus, NavigationStatusCreatorFn>("NavigationStatus");
    if (creator != nullptr)
    {
        mEndpoint = creator(mSessionId,
                            mReceiver->messageRouter(),
                            NAVI_INTERVAL_MS,
                            NAVI_IMAGE_HEIGHT,
                            NAVI_IMAGE_WIDTH,
                            NAVI_IMAGE_COLOR,
                            NavigationStatusService_InstrumentClusterType_ENUM);

    }
    else
    {
        LOG_ERROR((demo, "NaviStatusChannel::Initialize() No Creator function registered with  factory for NavigationStatus"));
        return false;
    }

    if(mEndpoint == nullptr)
    {
        LOG_ERROR((demo, "NaviStatusChannel::Initialize() failed to create  endpoint for NavigationStatus "));
        return false;
    }

    mEndpoint->registerCallbacks(this);

    if(!mEndpoint->init())
    {
        LOG_ERROR((demo, "failed to initialize  navigation status endpoint"));
        return false;
    }
    // The argument of API is done const_cast because GalReceiver requires non const pointer.
    if(!mReceiver->registerService(const_cast<AditNavigationStatus*>(mEndpoint.get())))
    {
        LOG_ERROR((demo, "failed to register navigation  status endpoint to receiver"));
        return false;
    }

    return true;
}
void NaviStatusChannel::navigationStatusCallback(int status)
{
    std::string str("");

    switch(status)
    {
    case NavigationStatus_NavigationStatusEnum_UNAVAILABLE:
        str += "UNAVAILABLE";
        break;
    case NavigationStatus_NavigationStatusEnum_ACTIVE:
        mSensor->sensorWork(SENSOR_LOCATION);     // Send location data by sensorSource module (Why Sensor used here ?)
        str += "ACTIVE";
        break;
    case NavigationStatus_NavigationStatusEnum_INACTIVE:
        mSensor->sensorBreak(SENSOR_LOCATION);    // Break sending of location data by sensorSource module
        str += "INACTIVE";
        break;
    default:
        str += "Error: Abnormal Navigation status";
        break;
    }
    LOGD_DEBUG((demo, "-- Navigation status callback (Status: %s) --", str.c_str()));
}

void NaviStatusChannel::navigationNextTurnCallback(const std::string &road, int turnSide, int event, const std::string &image, int turnAngle, int turnNumber)
{

    std::string turnSideStr[] = {"ABNORMAL_EVENT" ,"LEFT" ,"RIGHT" ,"UNSPECIFIED"};
    std::string eventStr[] = {"UNKNOWN" ,"DEPART" ,"NAME CHANGE" ,"SLIGHT_TURN" ,"TURN" ,"SHARP_TURN", "U_TURN" ,"ON_RAMP"
                         "OFF_RAMP" ,"FORK" ,"MERGE" ,"ROUNDABOUT_ENTER" ,"ROUNDABOUT_EXIT" ,"ROUNDABOUT_ENTER_AND_EXIT",
                         "STRAIGHT " ,"ABNORMAL_EVENT" ,"FERRY_BOAT" ,"FERRY_TRAIN" ,"DESTINATION"};
    std::
    string str =    "Road: " + road + " TurnSide: " + turnSideStr[turnSide] + " TurnAngle: " + std::to_string(turnAngle) + " TurnNumber: " + std::to_string(turnNumber);
    if(event < 19)
    {
        LOGD_DEBUG((demo, "-- Navigation Next Turn callback -- Event: %s(%d) %s", eventStr[event].c_str(), event, str.c_str()));
    }
    else
    {
        LOG_ERROR((demo, "-- Navigation Next Turn callback -- Event: %d %s", event, str.c_str()));
    }

}

void NaviStatusChannel::navigationNextTurnDistanceCallback(int distanceMeters, int timeSeconds, int displayDistanceE3,
                                                             NavigationNextTurnDistanceEvent_DistanceUnits displayDistanceUnit)
{
    LOGD_DEBUG((demo, "-- Navigation Next Turn distance callback (Distance:%dm time: %dsec, distance: %d, event: %d) --",
            distanceMeters, timeSeconds, displayDistanceE3, (int)displayDistanceUnit));
}


} } // namespace adit { namespace aauto
